/* =========================================
   IAFluence Dashboard — app.js v2
   Loads data.json and renders the full dashboard.
   To update content: edit data.json only.
   Charts: Chart.js 4.x  |  Icons: Lucide
========================================= */

'use strict';

// ── Utility helpers ──────────────────────────────────────────────
const $ = (sel, ctx = document) => ctx.querySelector(sel);
const $$ = (sel, ctx = document) => [...ctx.querySelectorAll(sel)];
const el = (tag, cls, html = '') => {
  const e = document.createElement(tag);
  if (cls) e.className = cls;
  if (html) e.innerHTML = html;
  return e;
};
const chipClass = v => v >= 8 ? 'chip hi' : v >= 5 ? 'chip md' : 'chip lo';
const trendIcon = t => ({ up: '↑', up2: '↑↑', down: '↓', stable: '→' }[t] || '→');
const trendColor = t => ({ up: 'var(--green)', up2: 'var(--green)', down: 'var(--red)', stable: 'var(--text-muted)' }[t] || 'var(--text-muted)');
const providerClass = p => ({ ChatGPT: 'chatgpt', Claude: 'claude', Gemini: 'gemini', DeepSeek: 'deepseek' }[p] || 'chatgpt');
const debounce = (fn, wait = 160) => {
  let t;
  return (...args) => { clearTimeout(t); t = setTimeout(() => fn(...args), wait); };
};

// ── FETCH & BOOT ─────────────────────────────────────────────────
fetch('data.json')
  .then(r => { if (!r.ok) throw new Error('Não foi possível carregar data.json'); return r.json(); })
  .then(data => render(data))
  .catch(err => {
    document.body.innerHTML = `<div style="font-family:sans-serif;padding:60px;text-align:center;color:#F32534;">
      <h2>⚠ Erro ao carregar data.json</h2><p style="color:#666;margin-top:12px;">${err.message}</p>
      <p style="color:#666;margin-top:8px;font-size:13px;">Certifique-se de que o arquivo data.json está na mesma pasta e que está sendo servido por um servidor HTTP (não file://).</p>
    </div>`;
  });

// ── MAIN RENDER ──────────────────────────────────────────────────
function render(D) {
  document.title = `${D.meta.brand} · ${D.meta.report_title}`;
  setText('#meta-brand', D.meta.brand);
  setText('#meta-category', D.meta.category);
  setText('#meta-period', D.meta.period);
  setText('#meta-version', D.meta.version);
  setText('#header-title', D.meta.report_title);
  setText('#header-subtitle', `${D.meta.brand} · ${D.meta.report_title} · ${D.meta.period}`);
  setText('#footer-period', D.meta.period);
  setText('#footer-version', D.meta.version);
  setText('#footer-period-b', D.meta.period);

  if (D.ui_text) injectUIText(D.ui_text);
  renderAuditoria(D.auditoria);
  renderScorecard(D.scorecard);
  renderResumo(D.resumo);
  renderSOV(D.sov);
  renderPresenca(D.presenca);
  renderPolaridade(D.polaridade, D.polaridade_alert);
  renderSOVProvedor(D.sov_provedor);
  renderArquetipo(D.arquetipo);
  renderAutoridade(D.autoridade);
  renderCoocorrencia(D.coocorrencia);
  renderConcorrentes(D.concorrentes);
  renderTemas(D.temas);
  renderDescricao(D.descricao_llm);
  renderForcas(D.forcas);
  renderPrompts(D.prompts);
  renderFontes(D.fontes);
  if (D.ai_traffic) renderAITraffic(D.ai_traffic);
  renderSEOTecnico(D.seo_tecnico);
  renderRoadmap(D.roadmap);

  setTimeout(() => {
    initChartSOVDonut(D.sov.donut);
    renderSovRanking(D.sov.donut.labels, D.sov.donut.values);
    initChartSOVTrend(D.sov.trend, D.concorrentes);
    if (D.sov.sov_por_ia) initChartSOVStackedBar(D.sov.sov_por_ia);
    initChartArchetype(D.arquetipo.chart);
    initChartCompetitorRadar(D.concorrentes.top_radar);
    buildGeoRanking(D.concorrentes.ranking);
    if (D.ai_traffic) {
      initChartAIPlatformsDonut(D.ai_traffic.donut);
      renderAIPlatformsLegend(D.ai_traffic);
    }
    initSidebarSubmenus();
    initMobileNav();
    initSectionTooltips();
    lucide.createIcons();
    initScroll();
    initFadeUp();
  }, 50);
}

// ── INJEÇÃO DE TEXTOS DA UI (ui_text do JSON) ────────────────────
function injectUIText(ui) {
  const map = {
    'auditoria':    { t: 'st-auditoria',    s: 'ss-auditoria',    b: 'sib-auditoria' },
    'scorecard':    { t: 'st-scorecard',    s: 'ss-scorecard',    b: 'sib-scorecard' },
    'resumo':       { t: 'st-resumo',       s: 'ss-resumo',       b: 'sib-resumo' },
    'sov':          { t: 'st-sov',          s: 'ss-sov',          b: 'sib-sov' },
    'sov_por_ia':   { t: 'st-sov-por-ia',   s: 'ss-sov-por-ia',   b: 'sib-sov-por-ia' },
    'sov_tendencia':{ t: 'st-sov-tendencia',s: 'ss-sov-tendencia', b: 'sib-sov-tendencia' },
    'sov_provedor': { t: 'st-sov-provedor', s: 'ss-sov-provedor',  b: 'sib-sov-provedor' },
    'presenca':     { t: 'st-presenca',     s: 'ss-presenca',     b: 'sib-presenca' },
    'polaridade':   { t: 'st-polaridade',   s: 'ss-polaridade',   b: 'sib-polaridade' },
    'arquetipo':    { t: 'st-arquetipo',    s: 'ss-arquetipo',    b: 'sib-arquetipo' },
    'autoridade':   { t: 'st-autoridade',   s: 'ss-autoridade',   b: 'sib-autoridade' },
    'concorrentes': { t: 'st-concorrentes', s: 'ss-concorrentes', b: 'sib-concorrentes' },
    'coocorrencia': { t: 'st-coocorrencia', s: 'ss-coocorrencia', b: 'sib-coocorrencia' },
    'temas':        { t: 'st-temas',        s: 'ss-temas',        b: 'sib-temas' },
    'descricao':    { t: 'st-descricao',    s: 'ss-descricao',    b: 'sib-descricao' },
    'forcas':       { t: 'st-forcas',       s: 'ss-forcas',       b: 'sib-forcas' },
    'prompts':      { t: 'st-prompts',      s: 'ss-prompts',      b: 'sib-prompts' },
    'fontes':       { t: 'st-fontes',       s: 'ss-fontes',       b: 'sib-fontes' },
    'ai_trafego':   { t: 'st-ai-trafego',   s: 'ss-ai-trafego',   b: 'sib-ai-trafego' },
    'seo_tecnico':  { t: 'st-seo-tecnico',  s: 'ss-seo-tecnico',  b: 'sib-seo-tecnico' },
    'roadmap':      { t: 'st-roadmap',      s: 'ss-roadmap',      b: 'sib-roadmap' },
  };
  Object.entries(map).forEach(([key, ids]) => {
    const block = ui[key];
    if (!block) return;
    if (block.title)       setText('#' + ids.t, block.title);
    if (block.description) setText('#' + ids.s, block.description);
    // Store tooltip text as data-tip on the button for JS-driven tooltip
    if (block.tooltip) {
      const btn = document.getElementById(ids.b);
      if (btn) btn.dataset.tip = block.tooltip;
    }
  });
}
// ── SECTION RENDERERS ────────────────────────────────────────────

// Task 3: Render audit compact card
function renderAuditoria(aud) {
  if (!aud) return;
  setText('#audit-last-date', aud.last_audit_date || aud.ultima_auditoria || '—');
  setText('#audit-next-date', aud.next_audit_date  || aud.proxima_auditoria || '—');
  setText('#audit-summary',   aud.summary || '');
  setText('#audit-note',      aud.note    || '');
  // Keep legacy fallback for any old IDs still present
  const band = $('#audit-info-band');
  if (band) {
    band.innerHTML = `
      <div class="audit-card highlight">
        <div class="audit-card-icon blue"><i data-lucide="calendar-check"></i></div>
        <div>
          <div class="audit-card-label">Última auditoria realizada</div>
          <div class="audit-card-value">${aud.last_audit_date || aud.ultima_auditoria || '—'}</div>
          <div class="audit-card-sub">Dados coletados e processados por ${aud.responsavel || 'IAFluence'}</div>
        </div>
      </div>
      <div class="audit-card">
        <div class="audit-card-icon green"><i data-lucide="calendar-clock"></i></div>
        <div>
          <div class="audit-card-label">Próxima auditoria agendada</div>
          <div class="audit-card-value">${aud.next_audit_date || aud.proxima_auditoria || '—'}</div>
          <div class="audit-card-sub">Faltam <strong>${aud.dias_para_proxima || '—'} dias</strong> para a próxima análise</div>
        </div>
      </div>
      <div class="audit-card">
        <div class="audit-card-icon yellow"><i data-lucide="file-badge"></i></div>
        <div>
          <div class="audit-card-label">Versão deste relatório</div>
          <div class="audit-card-value" id="audit-version">—</div>
          <div class="audit-card-sub">Período de referência: <span id="audit-period">—</span></div>
        </div>
      </div>
      <div class="audit-note-card">
        <i data-lucide="info"></i>
        <div class="audit-note-text">${aud.nota || aud.note || ''}</div>
      </div>
    `;
    setTimeout(() => {
      setText('#audit-version', document.getElementById('meta-version')?.textContent || '—');
      setText('#audit-period',  document.getElementById('meta-period')?.textContent  || '—');
    }, 80);
  }
}

function renderScorecard(sc) {
  // Score ring animation
  const circle = $('#score-ring-circle');
  const numEl  = $('#score-ring-num');
  const pctEl  = $('#score-percentile');
  const synEl  = $('#score-synthesis');

  if (circle) {
    const circ = 2 * Math.PI * 54;
    const pct  = sc.geo_score / sc.geo_max;
    setTimeout(() => {
      circle.style.strokeDasharray = `${circ * pct} ${circ * (1 - pct)}`;
    }, 200);
  }
  if (numEl)  numEl.textContent  = sc.geo_score;
  if (pctEl)  pctEl.textContent  = sc.percentile + 'º';
  if (synEl)  synEl.textContent  = sc.synthesis;

  // KPI grid
  const kpiGrid = $('#kpi-grid');
  if (kpiGrid && sc.kpis) {
    kpiGrid.innerHTML = sc.kpis.map(k => `
      <div class="kpi-card ${k.color}">
        <div class="kpi-icon ${k.color}"><i data-lucide="${k.icon}"></i></div>
        <div class="kpi-val">${k.value}</div>
        <div class="kpi-lbl">${k.label}</div>
        <div class="kpi-delta ${k.trend === 'up' ? 'up' : 'down'}">
          ${k.trend === 'up' ? '↑' : '↓'} ${k.delta}
        </div>
      </div>`).join('');
  }

  // Task 4: Segmentation cards in scorecard
  const segWrap = $('#scorecard-segmentation');
  if (segWrap && sc.segmentation) {
    segWrap.innerHTML = sc.segmentation.map(s => `
      <div class="seg-card">
        <div class="seg-val">${s.value}</div>
        <div class="seg-lbl">${s.label}</div>
        <div class="seg-detail">${s.detail}</div>
      </div>`).join('');
  }

  // Risk & Opportunity
  if (sc.risk) {
    setText('#score-risk-title', sc.risk.title);
    setText('#score-risk-text', sc.risk.text);
  }
  if (sc.opportunity) {
    setText('#score-opp-title', sc.opportunity.title);
    setText('#score-opp-text', sc.opportunity.text);
  }
}

function renderResumo(res) {
  setText('#resumo-desc', res.description);
  const m = res.maturity;
  setText('#resumo-geo',        m.geo_score + '/100');
  setText('#resumo-sentiment',  '+' + m.sentiment_index);
  setText('#resumo-consistency', m.consistency + '%');
  setText('#resumo-percentile',  m.percentile + 'º');
  const kws = $('#resumo-keywords');
  if (kws) kws.innerHTML = res.keywords.map(k => `<span class="arch-kw">${k}</span>`).join('');
}

function renderSOV(sov) {
  // Segmentation cards are now in scorecard (moved via data.json)
  // Nothing to render here for segmentation anymore
}

function renderSovRanking(labels, values) {
  const wrap = $('#sov-ranking');
  if (!wrap) return;
  const total = values.reduce((a, b) => a + b, 0) || 1;
  const items = labels.map((l, i) => ({ name: l, val: values[i] }));
  wrap.innerHTML = items.map((item, i) => `
    <div class="sov-rank-item${i === 0 ? ' is-you' : ''}">
      <div class="sov-rank-left">
        <div class="sov-rank-pos">${i + 1}</div>
        <div class="sov-rank-name">${item.name}${i === 0 ? '<span class="chip-you">Você</span>' : ''}</div>
      </div>
      <div class="sov-rank-val">${item.val}%</div>
    </div>`).join('');
}

function renderPresenca(providers) {
  const wrap = $('#presenca-grid');
  if (!wrap) return;
  wrap.innerHTML = providers.map(p => `
    <div class="provider-row fade-up">
      <div class="provider-header">
        <div class="provider-name-wrap">
          <div class="provider-icon"><img src="${p.icon}" alt="${p.provider}"></div>
          <div>
            <div class="provider-name">${p.provider}</div>
            <div class="provider-detail">${p.detail}</div>
          </div>
        </div>
        <div class="provider-pct" style="color:${p.color};">${p.pct}%</div>
      </div>
      <div class="provider-bar-track">
        <div class="provider-bar-fill" style="width:${p.pct}%;background:${p.color};"></div>
      </div>
      <div class="provider-tags">${p.tags.map(t => `<span class="provider-tag">${t}</span>`).join('')}</div>
      <div class="provider-factor">${p.factor}</div>
    </div>`).join('');
}

function renderPolaridade(rows, alert) {
  const wrap = $('#polaridade-table');
  if (!wrap) return;
  wrap.innerHTML = rows.map(r => `
    <div class="pol-row">
      <div class="pol-lbl">${r.provider}</div>
      <div class="pol-bars">
        <div class="pol-pos" style="width:${r.positive}%;"></div>
        <div class="pol-neu" style="width:${r.neutral}%;"></div>
        <div class="pol-neg" style="width:${r.negative}%;"></div>
      </div>
      <div class="pol-pct">${r.positive}% positivo</div>
    </div>`).join('');

  const alertEl = $('#polaridade-alert');
  if (alertEl && alert) {
    alertEl.innerHTML = `
      <i data-lucide="alert-triangle"></i>
      <div>
        <div class="alert-title">${alert.title}</div>
        <div class="alert-body">${alert.text}</div>
      </div>`;
  }
}

function renderSOVProvedor(providers) {
  const wrap = $('#sov-provedor-grid');
  if (!wrap) return;
  wrap.innerHTML = providers.map(p => `
    <div class="sov-card" style="border-color:${p.border};">
      <div class="sov-card-hd">
        <div class="sov-ico"><img src="${p.icon}" alt="${p.provider}"></div>
        <div class="sov-name" style="color:${p.color};">${p.provider}</div>
      </div>
      <div class="sov-val" style="color:#011A3E;">${p.pct}%</div>
      <div class="sov-delta" style="color:${p.trend === 'up' ? 'var(--sky)' : 'var(--red)'};">${p.delta} vs mês anterior</div>
      <div class="sov-stats" style="border-color:${p.border};">
        <div class="sov-stat"><span class="sov-stat-l">Marca</span><span class="sov-stat-v" style="color:#011A3E;">${p.stat_brand}</span></div>
        <div class="sov-stat"><span class="sov-stat-l">Categoria</span><span class="sov-stat-v" style="color:#011A3E;">${p.stat_cat}</span></div>
        <div class="sov-stat"><span class="sov-stat-l">Problema</span><span class="sov-stat-v" style="color:#011A3E;">${p.stat_prob}</span></div>
      </div>
    </div>`).join('');
}

function renderArquetipo(arq) {
  setText('#arch-name', arq.name);
  setText('#arch-desc', arq.description);

  const topCards = $('#arch-top-cards');
  if (topCards && arq.chart && arq.chart.labels && arq.chart.values) {
    const pairs = arq.chart.labels.map((l, i) => ({ label: l, value: arq.chart.values[i] }))
      .sort((a, b) => b.value - a.value);
    const icons = ['✦', '◆'];
    const ranks = ['1º arquétipo', '2º arquétipo'];
    topCards.innerHTML = [pairs[0], pairs[1]].map((p, i) => `
      <div class="arch-top-card ${i === 0 ? 'primary' : 'secondary'}">
        <span class="arch-top-rank">${ranks[i]}</span>
        <div class="arch-top-badge">${icons[i]}</div>
        <div>
          <div class="arch-top-name">${p.label}</div>
          <div class="arch-top-pct">${p.value}%</div>
        </div>
      </div>`).join('');
  }

  const kws = $('#arch-keywords');
  if (kws) kws.innerHTML = arq.keywords.map(k => `<span class="arch-kw">${k}</span>`).join('');

  const cons = $('#arch-consistency');
  if (cons) {
    cons.innerHTML = arq.consistency.map(c => `
      <div class="arch-llm-row">
        <div class="arch-llm-name">
          <img src="${c.icon}" alt="${c.provider}">
          ${c.provider}
        </div>
        <div style="display:flex;align-items:center;gap:8px;">
          <span class="arch-llm-val">${c.archetype}</span>
          ${c.warning ? '<span class="arch-warn">⚠ Divergência</span>' : ''}
        </div>
      </div>`).join('');
  }
}

function renderAutoridade(aut) {
  const providers = ['ChatGPT', 'Claude', 'Gemini', 'DeepSeek'];
  const tbody = $('#autoridade-tbody');
  if (!tbody) return;
  tbody.innerHTML = aut.territories.map(t => `
    <tr>
      <td class="territory-name">${t.name}</td>
      ${providers.map(p => {
        const v = t.scores[p];
        return `<td style="text-align:center;"><span class="${chipClass(v)}">${v}</span></td>`;
      }).join('')}
    </tr>`).join('');
}

function renderCoocorrencia(items) {
  const wrap = $('#coocorrencia-grid');
  if (!wrap) return;
  wrap.innerHTML = items.map(c => `
    <div class="cooc-card fade-up">
      <div class="cooc-hd">
        <div class="cooc-name">${c.name}</div>
        <span class="cooc-freq">${c.freq_label}</span>
      </div>
      <div class="cooc-bar-lbl">Frequência de co-ocorrência</div>
      <div class="cooc-bar-track"><div class="cooc-bar-fill" style="width:${c.frequency_pct}%;"></div></div>
      <div class="cooc-quote">${c.quote}</div>
      <div class="cooc-note">${c.note}</div>
    </div>`).join('');
}

function renderConcorrentes(comp) {
  renderConcorrentesTable(comp.table);
}

function renderConcorrentesTable(rows) {
  const tbody = $('#concorrentes-tbody');
  if (!tbody) return;
  tbody.innerHTML = rows.map(r => {
    const isYou = r.isYou;
    const chipGeo = r.geo >= 80 ? 'chip hi' : r.geo >= 60 ? 'chip md' : 'chip lo';
    const chipP = v => v >= 75 ? 'chip hi' : v >= 60 ? 'chip md' : 'chip lo';
    return `<tr${isYou ? ' style="background:rgba(33,51,253,0.04);border-left:3px solid var(--blue);"' : ''}>
      <td>
        <div style="font-weight:${isYou ? '800' : '700'};color:${isYou ? 'var(--blue)' : 'var(--dark)'};">${r.name}${isYou ? ' ★' : ''}</div>
        <div style="font-size:11px;color:var(--text-muted);">${r.country}</div>
      </td>
      <td style="text-align:center;"><span class="${chipGeo}" style="${isYou ? 'background:rgba(33,51,253,0.15);color:var(--blue);' : ''}">${r.geo}</span></td>
      <td style="text-align:center;"><span class="${chipP(r.chatgpt)}">${r.chatgpt}%</span></td>
      <td style="text-align:center;"><span class="${chipP(r.claude)}">${r.claude}%</span></td>
      <td style="text-align:center;"><span class="${chipP(r.gemini)}">${r.gemini}%</span></td>
      <td style="text-align:center;"><span class="${chipP(r.deepseek)}">${r.deepseek}%</span></td>
      <td style="text-align:center;font-weight:${isYou ? '800' : '700'};color:${isYou ? 'var(--blue)' : 'var(--dark)'};">${r.sov}</td>
      <td style="text-align:center;">
        <span style="font-size:11.5px;font-weight:700;padding:2px 8px;border-radius:20px;background:${r.sentiment >= 60 ? 'rgba(22,163,74,0.12)' : r.sentiment >= 40 ? 'rgba(255,199,0,0.12)' : 'rgba(243,37,52,0.09)'};color:${r.sentiment >= 60 ? 'var(--green)' : r.sentiment >= 40 ? '#8a6800' : 'var(--red)'};">+${r.sentiment}</span>
      </td>
    </tr>`;
  }).join('');
}

function renderTemas(temas) {
  const wrap = $('#temas-list');
  if (!wrap) return;
  wrap.innerHTML = temas.map(t => `
    <div class="theme-item fade-up">
      <div class="theme-name">${t.name}</div>
      <div class="theme-bar-wrap">
        <div class="theme-bar-track">
          <div class="theme-bar-fill" style="width:${t.pct}%;background:var(--blue);"></div>
        </div>
      </div>
      <div class="theme-count">${t.mentions} menções</div>
      <div class="theme-trend ${t.trend}">${t.trend_label}</div>
      <div class="theme-tags">${t.tags.map(tg => `<span class="theme-tag">${tg}</span>`).join('')}</div>
    </div>`).join('');
}

function renderDescricao(items) {
  const wrap = $('#descricao-grid');
  if (!wrap) return;
  wrap.innerHTML = items.map(d => `
    <div class="desc-card fade-up" style="border-color:${d.border};">
      <div class="desc-card-hd" style="border-color:${d.border};">
        <div class="desc-ico"><img src="${d.icon}" alt="${d.provider}"></div>
        <div class="desc-card-name" style="color:${d.color};">${d.provider}</div>
      </div>
      <div class="desc-quote">${d.quote}</div>
      <div class="desc-tags">${d.tags.map(tg => `<span class="desc-tag" style="background:${d.color}18;color:${d.color};">${tg}</span>`).join('')}</div>
    </div>`).join('');
}

function renderForcas(f) {
  const sWrap = $('#strengths-col');
  const aWrap = $('#attention-col');
  if (sWrap) sWrap.innerHTML = f.strengths.map(s => forcaCard(s, 'green')).join('');
  if (aWrap) aWrap.innerHTML = f.attention.map(s => forcaCard(s, 'red')).join('');
}

function forcaCard(item, type) {
  const badges = item.providers.map(p =>
    `<span class="llm-badge ${p}"><img src="assets/icons/${p}.png" alt="${p}">${p}</span>`
  ).join('');
  return `<div class="str-card ${type} fade-up">
    <div class="str-title">${item.title}</div>
    <div class="str-desc">${item.desc}</div>
    <div class="str-foot">
      <div class="llm-badges">${badges}</div>
      <span class="rec-badge">${item.recurrence} de recorrência</span>
    </div>
  </div>`;
}

function renderPrompts(prompts) {
  renderPromptSet('#prompts-dom', prompts.domina);
  renderPromptSet('#prompts-pres', prompts.aparece);
  renderPromptSet('#prompts-abs', prompts.ausente);
  const setCount = (id, arr) => {
    const e = document.getElementById(id);
    if (e && arr) e.textContent = arr.length;
  };
  setCount('count-dom', prompts.domina);
  setCount('count-pres', prompts.aparece);
  setCount('count-abs', prompts.ausente);
}

function renderPromptSet(selector, items) {
  const wrap = $(selector);
  if (!wrap) return;
  wrap.innerHTML = items.map(item => `
    <div class="prompt-query fade-up">
      <div class="prompt-query-header">
        <div class="prompt-title">${item.title || 'Prompt'}</div>
        <div class="prompt-query-text">${item.query}</div>
      </div>
      <div class="prompt-responses">
        ${item.responses.map(r => `
          <div class="prompt-resp">
            <div class="prompt-resp-provider">
              <img src="${r.icon}" alt="${r.provider}"><span>${r.provider}</span>
            </div>
            <div>${r.text}</div>
          </div>`).join('')}
      </div>
      ${item.action ? `<div class="prompt-action"><i data-lucide="lightbulb"></i><span>${item.action}</span></div>` : ''}
    </div>`).join('');
}

function renderFontes(fontes) {
  const wrap = $('#fontes-grid');
  if (!wrap) return;
  wrap.innerHTML = fontes.map(f => `
    <div class="source-card ${f.status} fade-up" style="border-color:${f.color};">
      <div class="source-icon" style="background:${f.color}15;color:${f.color};">
        <i data-lucide="${f.icon}"></i>
      </div>
      <div class="source-name">${f.name}</div>
      <div class="source-pct" style="color:${f.color};">${f.pct}%</div>
      <div class="source-bar-track">
        <div class="source-bar-fill" style="width:${f.pct}%;background:${f.color};"></div>
      </div>
      <div class="source-desc">${f.desc}</div>
      ${f.status === 'warn' ? '<div style="margin-top:8px;font-size:11px;font-weight:700;color:var(--yellow);">⚠ Requer ação</div>' : ''}
    </div>`).join('');
}

function renderAITraffic(at) {
  if (!at) return;
  const metaTxt = `Período: ${at.meta?.period || '—'} | Comparar com: ${at.meta?.compare || '—'} | Propriedade: ${at.meta?.property || '—'}`;
  setText('#ai-traffic-meta', metaTxt);
  setText('#ai-platforms-sub', metaTxt);
  setText('#ai-traffic-sub', metaTxt);

  const fmt = v => (typeof v === 'number') ? v.toLocaleString('pt-BR') : (v ?? '—');
  const cleanDelta = d => (d ?? '—').toString().replace(/^\+/, '').replace(/^-/, '');

  const tbody = $('#aiTrafficTbody');
  if (tbody) {
    tbody.innerHTML = '';
    (at.sources || []).forEach(row => {
      const tr = document.createElement('tr');
      const tdSrc = document.createElement('td');
      tdSrc.innerHTML = `<div class="ai-cell left"><div class="ai-src">${row.source || '—'}</div></div>`;
      tr.appendChild(tdSrc);
      const mkCell = (k) => {
        const m = row[k] || {};
        const trend = (m.trend || 'stable');
        const cls = trend === 'up' ? 'up' : trend === 'down' ? 'down' : 'stable';
        const arrow = trend === 'up' ? '↑' : trend === 'down' ? '↓' : '→';
        const td = document.createElement('td');
        td.setAttribute('data-col', k);
        td.innerHTML = `<div class="ai-cell"><div class="ai-val">${fmt(m.value)}</div><div class="ai-delta ${cls}">${arrow} ${cleanDelta(m.delta)}</div></div>`;
        return td;
      };
      tr.appendChild(mkCell('users'));
      tr.appendChild(mkCell('engagement'));
      tr.appendChild(mkCell('sessions'));
      tr.appendChild(mkCell('conversions'));
      tbody.appendChild(tr);
    });
  }

  const wrap = $('#aiTableWrap');
  const btns = $$('#aiMetricCarousel .ai-metric-btn');
  if (wrap && btns.length) {
    btns.forEach(b => {
      b.addEventListener('click', () => {
        btns.forEach(x => x.classList.remove('active'));
        b.classList.add('active');
        wrap.dataset.show = b.dataset.metric || 'sessions';
      });
    });
  }
}

function renderAIPlatformsLegend(at) {
  const legend = $('#aiPlatformsLegend');
  if (!legend || !at || !at.donut) return;
  const labels = at.donut.labels || [];
  const values = at.donut.values || [];
  const colors = at.donut.colors || [];
  const total = values.reduce((a, b) => a + (Number(b) || 0), 0) || 1;
  const getSourceRow = (name) => (at.sources || []).find(s => s.source === name);
  const pct = (v) => {
    const p = (v / total) * 100;
    return (Math.round(p * 10) / 10).toString().replace('.', ',') + '%';
  };
  const cleanDelta = d => (d ?? '—').toString().replace(/^\+/, '').replace(/^-/, '');

  legend.innerHTML = labels.map((name, i) => {
    const v = Number(values[i]) || 0;
    const c = colors[i] || 'var(--blue)';
    const row = getSourceRow(name) || {};
    const sess = row.sessions || {};
    const trend = (sess.trend || 'stable');
    const cls = trend === 'up' ? 'up' : trend === 'down' ? 'down' : 'stable';
    const arrow = trend === 'up' ? '↑' : trend === 'down' ? '↓' : '→';
    const delta = cleanDelta(sess.delta);
    return `
      <div class="ai-legend-item">
        <div class="ai-legend-left">
          <span class="ai-dot" style="background:${c};"></span>
          <span class="ai-legend-name">${name}</span>
        </div>
        <div class="ai-legend-right">
          <span class="ai-legend-pct">${pct(v)}</span>
          <span class="ai-legend-delta ${cls}">${arrow} ${delta}</span>
        </div>
      </div>`;
  }).join('') + `
    <div class="ai-legend-total">
      <span>Total</span>
      <span>${values.reduce((a, b) => a + (Number(b) || 0), 0).toLocaleString('pt-BR')}</span>
    </div>`;
}

// Task 9: SEO Técnico - health score + big numbers + thematic + expanded table
function renderSEOTecnico(S) {
  if (!S) return;
  if (S.meta) setText('#seo-tech-meta', S.meta);

  // Health Score + Big Numbers (Fix 2: CSS conic-gradient ring)
  const healthWrap = $('#seo-health-wrap');
  if (healthWrap && S.health) {
    const h = S.health;
    const scoreColor = h.score >= 80 ? 'var(--green)' : h.score >= 60 ? '#F97316' : 'var(--red)';
    healthWrap.innerHTML = `
      <div class="seo-score-ring-wrap">
        <div class="seo-score-ring-label">Saúde do site</div>
        <div class="seo-ring-circle" style="--ring-pct:${h.score};--ring-color:${scoreColor};">
          <div class="seo-ring-inner">
            <div class="seo-ring-number" style="color:${scoreColor};">${h.score}%</div>
            <div class="seo-ring-pct-label">Saúde</div>
          </div>
        </div>
        <div class="seo-score-ring-sub">de ${h.pages_crawled.toLocaleString('pt-BR')} páginas rastreadas</div>
      </div>
      <div class="seo-bignumbers">
        <div class="seo-bignum-card">
          <div class="seo-bignum-val" style="color:var(--blue);">${h.pages_crawled.toLocaleString('pt-BR')}</div>
          <div class="seo-bignum-lbl">Páginas rastreadas</div>
        </div>
        <div class="seo-bignum-card">
          <div class="seo-bignum-val" style="color:var(--green);">${h.pages_healthy.toLocaleString('pt-BR')}</div>
          <div class="seo-bignum-lbl">Páginas íntegras</div>
        </div>
        <div class="seo-bignum-card">
          <div class="seo-bignum-val" style="color:#F97316;">${h.pages_issues.toLocaleString('pt-BR')}</div>
          <div class="seo-bignum-lbl">Com problemas</div>
        </div>
        <div class="seo-bignum-card">
          <div class="seo-bignum-val" style="color:var(--red);">${h.pages_broken.toLocaleString('pt-BR')}</div>
          <div class="seo-bignum-lbl">Quebradas</div>
        </div>
        <div class="seo-bignum-card">
          <div class="seo-bignum-val" style="color:var(--text-muted);">${h.pages_blocked.toLocaleString('pt-BR')}</div>
          <div class="seo-bignum-lbl">Bloqueadas</div>
        </div>
      </div>`;
  }

  // Thematic Scores
  const thematicWrap = $('#seo-thematic-wrap');
  if (thematicWrap && S.health && S.health.thematic_scores) {
    thematicWrap.innerHTML = `
      <div class="seo-thematic-title">Pontuação temática</div>
      <div class="seo-thematic-grid">
        ${S.health.thematic_scores.map(ts => {
          const c = ts.color;
          const scoreText = ts.score >= 80 ? 'Bom' : ts.score >= 60 ? 'Regular' : 'Crítico';
          return `
          <div class="seo-theme-card">
            <div class="seo-theme-header">
              <div class="seo-theme-name-wrap">
                <div class="seo-theme-icon" style="background:${c}15;color:${c};">
                  <i data-lucide="${ts.icon}"></i>
                </div>
                <div class="seo-theme-name">${ts.label}</div>
              </div>
              <div class="seo-theme-score" style="color:${c};">${ts.score}%</div>
            </div>
            <div class="seo-theme-bar-track">
              <div class="seo-theme-bar-fill" style="width:${ts.score}%;background:${c};"></div>
            </div>
          </div>`;
        }).join('')}
      </div>`;
  }

  // Errors table (Task 9: expanded columns)
  const tbody = $('#seo-tech-tbody');
  if (!tbody) return;
  tbody.innerHTML = '';
  const items = Array.isArray(S.items) ? S.items : [];
  items.forEach(it => {
    const tr = document.createElement('tr');
    const s = String(it.severity || '').toLowerCase();
    const isHigh = (s === 'alta' || s === 'high' || s === 'critica' || s === 'crítica');

    // Title col
    const td1 = document.createElement('td');
    td1.innerHTML = `<div class="seo-td-title">${it.title || '—'}</div>`;

    // Severity col
    const td2 = document.createElement('td');
    td2.className = 'tc';
    const sev = document.createElement('span');
    sev.className = 'sev ' + (isHigh ? 'high' : 'low');
    sev.textContent = isHigh ? 'Alta' : 'Média';
    td2.appendChild(sev);

    // Summary col
    const td3 = document.createElement('td');
    td3.innerHTML = `<div class="seo-td-text">${it.summary || it.problem || '—'}</div>`;

    // Why fix col
    const td4 = document.createElement('td');
    td4.innerHTML = `<div class="seo-td-text">${it.why_fix || it.impact || '—'}</div>`;

    // How fix col
    const td5 = document.createElement('td');
    td5.innerHTML = `<div class="seo-td-text">${it.how_fix || it.fix || it.how_to_fix || '—'}</div>`;

    tr.appendChild(td1);
    tr.appendChild(td2);
    tr.appendChild(td3);
    tr.appendChild(td4);
    tr.appendChild(td5);
    tbody.appendChild(tr);
  });
}

function renderRoadmap(rm) {
  const wrap = $('#roadmap-grid');
  if (!wrap) return;
  wrap.innerHTML = rm.columns.map(col => `
    <div>
      <div class="rm-col-hd ${col.id}" style="background:${col.color}12;color:${col.color};">
        <i data-lucide="${col.icon}"></i>${col.label}
      </div>
      ${col.items.map(item => `
        <div class="rm-item ${col.id} fade-up" style="border-color:${col.color};border-left-color:${col.color};">
          <div class="rm-title">${item.title}</div>
          <div class="rm-summary">${item.summary}</div>
          <div class="rm-tags">${item.tags.map(t => `<span class="rm-tag ${t}">${t}</span>`).join('')}</div>
          <div class="rm-impact"><i data-lucide="trending-up"></i>${item.impact}</div>
        </div>`).join('')}
    </div>`).join('');

  setText('#roadmap-summary-title', rm.summary.title);
  setText('#roadmap-summary-text', rm.summary.text);
}


// ── CHARTS ───────────────────────────────────────────────────────

// Task 11: AI Platforms Donut colors aligned with SOV colors
function initChartAIPlatformsDonut(d) {
  const ctx = document.getElementById('aiPlatformsDonut');
  if (!ctx || !d) return;
  // Use SOV-aligned colors for consistency
  const alignedColors = ['#2133FD', '#00B3FF', '#FE368C', '#FFC700'];
  new Chart(ctx.getContext('2d'), {
    type: 'doughnut',
    data: {
      labels: d.labels,
      datasets: [{ data: d.values, backgroundColor: alignedColors, borderWidth: 0, hoverOffset: 8 }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '68%',
      plugins: { legend: { display: false } },
      animation: { animateRotate: true, duration: 1200 }
    }
  });
}

function initChartSOVDonut(d) {
  const ctx = document.getElementById('sovDonut');
  if (!ctx) return;
  new Chart(ctx.getContext('2d'), {
    type: 'doughnut',
    data: {
      labels: d.labels,
      datasets: [{ data: d.values, backgroundColor: d.colors, borderWidth: 0, hoverOffset: 8 }]
    },
    options: {
      cutout: '68%',
      plugins: { legend: { display: false } },
      animation: { animateRotate: true, duration: 1200 }
    }
  });
}

// Task 6: Stacked bar chart - SOV per brand per AI
function initChartSOVStackedBar(sovPorIA) {
  const ctx = document.getElementById('sovStackedBar');
  if (!ctx || !sovPorIA) return;

  const brands    = sovPorIA.brands;    // eixo X — 11 marcas
  const providers = sovPorIA.providers; // grupos — 4 IAs

  // Cores definidas pelo cliente
  const providerColors = {
    'ChatGPT':  '#011A3E',
    'Claude':   '#FF7A02',
    'Gemini':   '#FFC700',
    'DeepSeek': '#00B3FF',
  };

  const datasets = providers.map(provider => {
    const c = providerColors[provider] || '#ccc';
    return {
      label: provider,
      data: brands.map(brand =>
        (sovPorIA.data[brand] ? sovPorIA.data[brand][providers.indexOf(provider)] : 0)
      ),
      backgroundColor: c,
      borderColor: c,
      borderWidth: 0,
      borderRadius: 4,
      borderSkipped: false,
      barPercentage: 0.80,
      categoryPercentage: 0.55,  // mais espaço entre grupos → marcas ficam mais largas
    };
  });

  // 120px per brand gives enough room for 4 bars + readable labels
  const minW = Math.max(brands.length * 120, ctx.parentElement.offsetWidth || 700);
  ctx.style.minWidth = minW + 'px';
  ctx.style.height   = '100%';

  new Chart(ctx.getContext('2d'), {
    type: 'bar',
    data: { labels: brands, datasets },
    options: {
      responsive: false,
      maintainAspectRatio: false,
      interaction: { mode: 'index', intersect: false },
      plugins: {
        legend: {
          position: 'bottom',
          labels: {
            font: { family: 'Epilogue', size: 12, weight: '700' },
            usePointStyle: true,
            pointStyle: 'rectRounded',
            boxWidth: 12,
            boxHeight: 10,
            padding: 18,
          }
        },
        tooltip: {
          backgroundColor: 'rgba(1,26,62,0.92)',
          titleFont: { family: 'Epilogue', size: 12, weight: '800' },
          bodyFont:  { family: 'Epilogue', size: 12 },
          padding: 12,
          cornerRadius: 8,
          callbacks: {
            title: (items) => items[0].label,
            label: (c) => `  ${c.dataset.label}: ${c.parsed.y.toFixed(1)}%`
          }
        }
      },
      scales: {
        x: {
          stacked: false,
          grid: { display: false },
          ticks: {
            font: { family: 'Epilogue', size: 11, weight: '700' },
            color: '#011A3E',
            maxRotation: 30,
            minRotation: 0,
            autoSkip: false,
          }
        },
        y: {
          stacked: false,
          beginAtZero: true,
          grid: { color: 'rgba(1,26,62,0.06)' },
          ticks: {
            font: { family: 'Epilogue', size: 11 },
            color: 'rgba(1,26,62,0.55)',
            callback: v => v + '%'
          }
        }
      },
      animation: { duration: 900 }
    }
  });
}
function initChartSOVTrend(t, concorrentes) {
  const ctx = document.getElementById('sovTrend');
  if (!ctx) return;
  const months = (t && Array.isArray(t.months)) ? t.months : [];
  const series = (t && Array.isArray(t.series)) ? t.series : [];
  const flat = [];
  series.forEach(s => (s.data || []).forEach(v => flat.push(v)));
  const minV = flat.length ? Math.min(...flat) : 0;
  const maxV = flat.length ? Math.max(...flat) : 40;

  new Chart(ctx.getContext('2d'), {
    type: 'line',
    data: {
      labels: months,
      datasets: series.map(s => ({
        label: s.label,
        data: s.data,
        borderColor: s.color,
        backgroundColor: s.fill ? (s.color + '14') : 'transparent',
        tension: 0.35,
        fill: !!s.fill,
        borderWidth: s.fill ? 2.8 : 1.6,
        pointRadius: s.fill ? 3.5 : 0,
        pointHoverRadius: s.fill ? 5 : 3,
        pointBackgroundColor: s.color,
        borderDash: s.dashed ? [4, 3] : []
      }))
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: { mode: 'nearest', intersect: false },
      plugins: {
        legend: { position: 'bottom', labels: { font: { family: 'Epilogue', size: 11 }, usePointStyle: true, boxWidth: 8, padding: 12 } },
        tooltip: { callbacks: { label: (c) => `${c.dataset.label}: ${c.parsed.y}%` } }
      },
      scales: {
        y: { beginAtZero: false, suggestedMin: Math.floor(minV - 2), suggestedMax: Math.ceil(maxV + 2), grid: { color: 'rgba(1,26,62,0.05)' }, ticks: { font: { family: 'Epilogue' }, callback: v => v + '%' } },
        x: { grid: { display: false }, ticks: { font: { family: 'Epilogue' }, maxRotation: 0, autoSkip: true } }
      }
    }
  });
}

function initChartArchetype(c) {
  const ctx = document.getElementById('archetypeChart');
  if (!ctx) return;
  new Chart(ctx.getContext('2d'), {
    type: 'bar',
    data: {
      labels: c.labels,
      datasets: [{
        label: 'Peso (%)',
        data: c.values,
        backgroundColor: c.colors,
        borderRadius: 6,
        borderSkipped: false
      }]
    },
    options: {
      indexAxis: 'y',
      plugins: {
        legend: { display: false },
        tooltip: { callbacks: { label: ctx => ' ' + ctx.parsed.x + '% de predominância' } }
      },
      scales: {
        x: { max: 38, grid: { color: 'rgba(1,26,62,0.05)' }, ticks: { font: { family: 'Epilogue', size: 11 }, callback: v => v + '%' } },
        y: { grid: { display: false }, ticks: { font: { family: 'Epilogue', size: 11, weight: '600' } } }
      },
      animation: { duration: 1200 }
    }
  });
}

function initChartCompetitorRadar(datasets) {
  const ctx = document.getElementById('competitorRadar');
  if (!ctx) return;
  const isMobile = window.matchMedia && window.matchMedia('(max-width: 900px)').matches;
  const isTiny   = window.matchMedia && window.matchMedia('(max-width: 560px)').matches;
  const chart = new Chart(ctx.getContext('2d'), {
    type: 'radar',
    data: {
      labels: ['SoV médio', 'ChatGPT', 'Claude', 'Gemini', 'DeepSeek', 'Sentimento'],
      datasets: datasets.map((ds, i) => ({
        label: ds.label,
        data: ds.data,
        borderColor: ds.color,
        backgroundColor: ds.color + (i === 0 ? '1e' : '0d'),
        borderWidth: ds.isYou ? (isMobile ? 2.2 : 2.5) : (isMobile ? 1.6 : 2),
        pointBackgroundColor: ds.color,
        pointRadius: ds.isYou ? (isMobile ? 4 : 5) : (isMobile ? 3 : 4)
      }))
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      layout: { padding: isMobile ? (isTiny ? 22 : 18) : 10 },
      scales: {
        r: {
          min: 0, max: 100,
          ticks: { stepSize: 20, display: !isMobile, font: { family: 'Epilogue', size: isMobile ? 10 : 11 }, color: 'rgba(1,26,62,0.5)', backdropColor: 'rgba(0,0,0,0)', showLabelBackdrop: false },
          grid: { color: 'rgba(1,26,62,0.1)' },
          pointLabels: {
            font: { family: 'Epilogue', size: isMobile ? (isTiny ? 10 : 11) : 12, weight: '600' },
            color: '#011A3E',
            padding: isMobile ? (isTiny ? 14 : 12) : 6,
            callback: (label) => {
              if (!isTiny) return label;
              if (label === 'SoV médio') return ['SoV', 'médio'];
              return label;
            }
          }
        }
      },
      plugins: {
        legend: {
          display: !isMobile,
          position: 'bottom',
          labels: { font: { family: 'Epilogue', size: 12 }, padding: 14, usePointStyle: true, boxWidth: 10 }
        }
      },
      animation: isMobile ? { duration: 0 } : { duration: 900 }
    }
  });
  const kick = () => { try { chart.resize(); chart.update('none'); } catch (e) {} };
  setTimeout(kick, 140);
  setTimeout(kick, 520);
  window.addEventListener('resize', debounce(kick, 180), { passive: true });
  window.addEventListener('orientationchange', () => setTimeout(kick, 220), { passive: true });
}

// Task 10: Light/friendly GEO Ranking
function buildGeoRanking(ranking) {
  const container = document.getElementById('geo-ranking');
  if (!container) return;
  container.innerHTML = '';
  const sorted = [...ranking].sort((a, b) => b.geo - a.geo);
  sorted.forEach((c, i) => {
    const div = document.createElement('div');
    div.className = 'geo-rank-row';
    div.innerHTML =
      `<span class="geo-rank-pos">${i + 1}</span>` +
      `<div class="geo-rank-bar-wrap"><div class="geo-rank-bar-fill" style="width:${c.geo}%;background:${c.color};"></div></div>` +
      `<span class="geo-rank-name${c.isYou ? ' is-you' : ''}">${c.name}${c.isYou ? ' ★' : ''}</span>` +
      `<span class="geo-rank-score" style="color:${c.color};">${c.geo}</span>`;
    container.appendChild(div);
  });
}


// ── SECTION INFO TOOLTIPS — global singleton approach ────────────
// A single floating div is repositioned on hover; no tooltip is nested
// inside a button, which eliminates all mouseleave flicker issues.
function initSectionTooltips() {
  const tip = document.getElementById('global-section-tip');
  if (!tip) return;

  // Build lookup: sib-KEY → tooltip text (from stt-KEY elements that no longer exist)
  // Instead we read directly from the injected ui_text already stored in data
  // The button IDs follow the pattern sib-{key} and data is already on the DOM
  // We'll store the text as data-tip on each button during injectUIText

  let hideTimer = null;

  const show = (btn) => {
    clearTimeout(hideTimer);
    const text = btn.dataset.tip;
    if (!text) return;

    tip.textContent = text;

    const rect = btn.getBoundingClientRect();
    const margin = 12;
    const tipW   = 280;

    // Horizontal: align arrow (left: 18px inside box) under the button centre
    let left = rect.left + rect.width / 2 - 18;
    // Clamp to viewport
    left = Math.max(margin, Math.min(left, window.innerWidth - tipW - margin));

    // Vertical: appear below the button
    const top = rect.bottom + 10;

    tip.style.left = left + 'px';
    tip.style.top  = top  + 'px';

    // Reposition arrow: offset so it points at button centre
    const arrowLeft = (rect.left + rect.width / 2) - left - 5;
    tip.style.setProperty('--arrow-left', Math.max(8, Math.min(arrowLeft, tipW - 18)) + 'px');

    tip.classList.add('visible');
  };

  const hide = () => {
    hideTimer = setTimeout(() => tip.classList.remove('visible'), 80);
  };

  $$('.section-info-btn').forEach(btn => {
    btn.addEventListener('mouseenter', () => show(btn));
    btn.addEventListener('mouseleave', hide);
    btn.addEventListener('focus',      () => show(btn));
    btn.addEventListener('blur',       hide);
  });
}

// ── SIDEBAR SUBMENUS (Task 8) ─────────────────────────────────────
function initSidebarSubmenus() {
  const toggles = $$('.nav-group-toggle');
  toggles.forEach(btn => {
    btn.addEventListener('click', () => {
      const expanded = btn.getAttribute('aria-expanded') === 'true';
      const groupId  = btn.dataset.group;
      const items    = document.getElementById('group-' + groupId);
      if (!items) return;
      btn.setAttribute('aria-expanded', String(!expanded));
      items.classList.toggle('collapsed', expanded);
    });
  });
}

// ── MOBILE NAV GROUPS (Fix 4) ──────────────────────────────────
function initMobileNav() {
  const groupBtns = $$('.mnav-group-btn');
  const panel     = $('#mobileNavPanel');
  const panelGroups = $$('.mnav-panel-group');
  const links     = $$('.mnav-panel-link');
  if (!panel || !groupBtns.length) return;

  groupBtns.forEach(btn => {
    btn.addEventListener('click', () => {
      const groupId   = btn.dataset.mgroup;
      const wasActive = btn.classList.contains('active');

      // Fechar tudo
      groupBtns.forEach(b => b.classList.remove('active'));
      panelGroups.forEach(g => g.classList.remove('active'));

      if (wasActive) {
        panel.classList.remove('open');
      } else {
        btn.classList.add('active');
        const target = document.querySelector(`.mnav-panel-group[data-panel="${groupId}"]`);
        if (target) target.classList.add('active');
        panel.classList.add('open');
      }
    });
  });

  links.forEach(link => {
    link.addEventListener('click', () => {
      panel.classList.remove('open');
      groupBtns.forEach(b => b.classList.remove('active'));
      panelGroups.forEach(g => g.classList.remove('active'));
    });
  });
}


// ── TAB SYSTEM ───────────────────────────────────────────────────
function showTab(id) {
  document.querySelectorAll('.pcon').forEach(e => e.classList.remove('active'));
  document.querySelectorAll('.ptab').forEach(b => b.classList.remove('active'));
  const content = document.getElementById('tab-' + id);
  if (content) content.classList.add('active');
  const btn = document.querySelector('.ptab[data-tab="' + id + '"]');
  if (btn) btn.classList.add('active');
}


// ── SCROLL & FADE ─────────────────────────────────────────────────
function initFadeUp() {
  const obs = new IntersectionObserver(entries => {
    entries.forEach(e => { if (e.isIntersecting) e.target.classList.add('visible'); });
  }, { threshold: 0.05 });
  document.querySelectorAll('.fade-up').forEach(e => obs.observe(e));
}

function initScroll() {
  const sections = document.querySelectorAll('section[id]');
  const navItems = document.querySelectorAll('.nav-item');
  window.addEventListener('scroll', () => {
    let current = '';
    sections.forEach(s => { if (window.scrollY >= s.offsetTop - 120) current = s.getAttribute('id'); });
    navItems.forEach(item => {
      item.classList.remove('active');
      if (item.getAttribute('href') === '#' + current) item.classList.add('active');
    });
  }, { passive: true });
}


// ── HELPERS ───────────────────────────────────────────────────────
function setText(sel, val) {
  const e = document.querySelector(sel);
  if (e) e.textContent = val;
}
